// ---------------- Vaunix LPS library example program ---------------
//
// Example program to demonstrate the use of the Vaunix library
// for the Vaunix Lab Brick Phase Shifters
// (c) 2015
//
// RD 9/24/15

#include <stdbool.h>	/* AK: Added include for 'bool' type */
#include <stdio.h>
#include <unistd.h>   	/* AK: Added include for error-free getlogin(). */
#include "LPShid.h"
#include <math.h>

#define FALSE 0
#define TRUE !FALSE

#define RUN_PROFILE 1	// set to 0 to skip the profile example

/* function prototypes */
void profileSine(DEVID deviceID, int phasemax);
void profileTriangle(DEVID deviceID, int phasemax);
void profileShow(int height);

/* globals */
int profile[VNX_PROFILE_LENGTH]; /* storage for phase profile calculation */


/* code begins here */
int main (int argc, char *argv[]) {
  int nDevices, nActive;
  int i, j, result;
  LPSTATUS status;
  int lastphase[MAXDEVICES];
  int waittime;
  int powerlevel;
  char cModelName[32];
  char c;
  char *username;
  DEVID activeDevices[MAXDEVICES];
  bool realhardware;

  /* AK: Added <unistd.h> to includes to avoid seg fault on getlogin(). */
  username = getlogin(); 
  printf("Hi %s,\r\n", username);

  if (0 != strcmp(username, "root")) {
    printf("Hi %s,\r\n", username);
    printf("Accessing USB ports on a Linux machine may require root level\r\n");
    printf("access. You are not logged in as root. You may be able to\r\n");
    printf("proceed if you have used 'chmod' to change the access mode\r\n");
    printf("of the appropriate devices in /dev/bus/usb. That requires\r\n");
    printf("root access also. We'll continue, but if you don't see your\r\n");
    printf("LPS devices or no data can be read from or written to them,\r\n");
    printf("that's probably the problem. su to root and try again.\r\n\r\n");
    printf("Try running with 'sudo', or become root by running 'su' before.\r\n\r\n");
	printf("You can use udev rules to allow access to USB devices by user processes.\r\n\r\n");
    
  }
  fnLPS_Init();
  /* If you have actual hardware attached, set this to TRUE. Setting to FALSE will run in test mode */
  realhardware = TRUE;
  fnLPS_SetTestMode(!realhardware);
  nDevices = fnLPS_GetNumDevices();
  printf("LPS test/demonstration program using library version %s\r\n\r\n", fnLPS_LibVersion());
  if (0 == nDevices) {
    printf("No Vaunix LPS devices located. Would you like to run in test mode? "); fflush(0);
    c = getchar();
    if ('Y' == (c & 0xdf)) {
      printf("\r\nSwitching to test mode.\r\n");
      realhardware = FALSE;
      fnLPS_Init();
      fnLPS_SetTestMode(!realhardware);
      nDevices = fnLPS_GetNumDevices();
    }
  }
  printf("Found %d devices\r\n", nDevices);

  for (i=1; i<=nDevices; i++) {
    result = fnLPS_GetModelName(i, cModelName);
    printf("  Model %d is %s (%d chars)\r\n", i, cModelName, result);
  }
  printf("\r\n");
  
  nActive = fnLPS_GetDevInfo(activeDevices);
  printf("We have %d active devices\r\n", nActive);

  for (i=0; i<nActive; i++) {
    /* let's open and init each device to get the threads running */
    status = fnLPS_InitDevice(activeDevices[i]);
    printf("  Opened device %d of %d. Return status=0x%08x (%s)\r\n", activeDevices[i], nActive, status, fnLPS_perror(status));
  }

  /* the data structure is filled by polling and we need a few seconds to do that */

  for (i=0; i<nActive; i++) {
    if (i > 0) printf("\r\n");

    /* only do this if not in test mode */
    printf("  Device %d is active\r\n", activeDevices[i]);
    /* dump what we know - that we read from the hardware */
    if (realhardware) {
	printf("  GetWorkingFrequency returned %d\r\n", fnLPS_GetWorkingFrequency(activeDevices[i]));
	printf("  GetPhaseAngle returned %d\r\n", fnLPS_GetPhaseAngle(activeDevices[i]));
	  
	printf("  -------- Ramp Parameters ---------\r\n");
	printf("  GetRampStart returned %d\r\n", fnLPS_GetRampStart(activeDevices[i]));
	printf("  GetRampEnd returned %d\r\n", fnLPS_GetRampEnd(activeDevices[i]));
	printf("  GetDwellTime returned %d\r\n", fnLPS_GetDwellTime(activeDevices[i]));
	printf("  GetDwellTimeTwo returned %d\r\n", fnLPS_GetDwellTimeTwo(activeDevices[i]));
	printf("  GetIdleTime returned %d\r\n", fnLPS_GetIdleTime(activeDevices[i]));
	printf("  GetHoldTime returned %d\r\n", fnLPS_GetHoldTime(activeDevices[i]));
	printf("  GetPhaseAngleStep returned %d\r\n", fnLPS_GetPhaseAngleStep(activeDevices[i]));
	printf("  GetPhaseAngleStepTwo returned %d\r\n", fnLPS_GetPhaseAngleStepTwo(activeDevices[i]));
	  
	printf("  -------- Profile Parameters ---------\r\n");
	printf("  GetProfileCount returned %d\r\n", fnLPS_GetProfileCount(activeDevices[i]));
	printf("  GetProfileDwellTime returned %d\r\n", fnLPS_GetProfileDwellTime(activeDevices[i]));
	printf("  GetProfileIdleTime returned %d\r\n", fnLPS_GetProfileIdleTime(activeDevices[i]));	  

	printf("  -------- Device Characteristics ---------\r\n");
	printf("  GetMinPhaseStep returned %d\r\n", fnLPS_GetMinPhaseStep(activeDevices[i]));
	printf("  GetMaxPhaseShift returned %d\r\n", fnLPS_GetMaxPhaseShift(activeDevices[i]));
	printf("  GetMinPhaseShift returned %d\r\n", fnLPS_GetMinPhaseShift(activeDevices[i]));
    }
    
    status = fnLPS_GetModelName(activeDevices[i], cModelName);
    printf("  Device %d (%s) has ", activeDevices[i], cModelName);
    status = fnLPS_GetSerialNumber(activeDevices[i]);
    printf("  Serial number=%d\r\n", status);

	// set the phase angle to some example values

  
    printf("0 degree phase angle for 5 seconds...\r\n");
    status = fnLPS_SetPhaseAngle(activeDevices[i], 0);
	if (status != STATUS_OK) printf ("SetPhaseAngle returned %s \r\n", fnLPS_perror(status));
    sleep(5);
	
    printf("45 degree phase angle for 5 seconds...\r\n");
    status = fnLPS_SetPhaseAngle(activeDevices[i], 45);
	if (status != STATUS_OK) printf ("SetPhaseAngle returned %s \r\n", fnLPS_perror(status));
    sleep(5);
	
    printf("90 degree phase angle for 5 seconds...\r\n");
    status = fnLPS_SetPhaseAngle(activeDevices[i], 90);
	if (status != STATUS_OK) printf ("SetPhaseAngle returned %s \r\n", fnLPS_perror(status));
    sleep(5);
	
    printf("180 degree phase angle for 5 seconds...\r\n");
    status = fnLPS_SetPhaseAngle(activeDevices[i], 180);
	if (status != STATUS_OK) printf ("SetPhaseAngle returned %s \r\n", fnLPS_perror(status));
    sleep(5);
	
    printf("270 degree phase angle for 5 seconds...\r\n");
    status = fnLPS_SetPhaseAngle(activeDevices[i], 270);
	if (status != STATUS_OK) printf ("SetPhaseAngle returned %s \r\n", fnLPS_perror(status));
    sleep(5);

  } // end of for loop over devices

#if RUN_PROFILE


	// for every other device, alternate between a sine wave and a triangle wave profile
	for (i=0; i<nActive; i++) {
      if (0 == i%2)
    	profileSine(activeDevices[i], fnLPS_GetMaxPhaseShift(activeDevices[i]));
      else
    	profileTriangle(activeDevices[i], fnLPS_GetMaxPhaseShift(activeDevices[i]));
	}
	
	// set the dwell time for each element in the profile to 100ms
	for (i=0; i<nActive; i++) {
	  status = fnLPS_SetProfileDwellTime(activeDevices[i], 100);
	  printf("Set profile dwell time for device %d status=0x%08x (%s)\r\n", activeDevices[i], status, fnLPS_perror(status));
	}	
    // now start the profiles
	for (i=0; i<nActive; i++) {
	  status = fnLPS_StartProfile(activeDevices[i], LPS_PROFILE_REPEAT);
	  printf("Started profile for device %d status=0x%08x (%s)\r\n", activeDevices[i], status, fnLPS_perror(status));
	}

	// show some of the phase angles as they change 
	for (i=0; i<nActive; i++){
	  lastphase[i] = -1;
	}
	waittime = 200;		// show output for 10 seconds, we'll check the phase angle about every .050 seconds
	while (waittime){
	  for (i=0; i<nActive; i++){
		result = fnLPS_GetPhaseAngle(activeDevices[i]);
		if (lastphase[i] != result){
			lastphase[i] = result;
			printf("Device %d has Phase Angle %d \r\n", activeDevices[i], result);
		}
	  }
		usleep(50000);	// wait for around 50 milliseconds
		waittime--;
	}

#endif

  /* close the devices */
  for (i=0; i<nActive; i++) {
    status = fnLPS_CloseDevice(activeDevices[i]);
    printf("Closed device %d. Return status=0x%08x (%s)\r\n", activeDevices[i], status, fnLPS_perror(status));
  }
  printf("End of test\r\n");
  return 0;
}

/* support functions */
void profileSine(DEVID deviceID, int phasemax) {
  /* calculate values for a sine wave phase shift profile. Use the size of
     the longest LPS profile and divide a full wave into that many segments. */
  int i, nsegs;
  float fi, fstart, fend, fstep;
  float ftemp;
  LPSTATUS status;
  
  nsegs = VNX_PROFILE_LENGTH;
  printf("Making a sine wave in %d segments\r\n", nsegs);
  fstart = 0;
  fend = 2.0 * M_PI; /* 2 PI = 1 whole circle */
  fstep = (fend - fstart) / (float)nsegs;
  fi = fstart;
  for (i=0; i<nsegs; i++) {
    /* sin() results range from -1.0 to +1.0, and we want to rescale this
       to 0.0 to 1.0 */ 
    ftemp = (1.0 + sin(fi)) / 2.0;
    /* and now that we have a 0-1 value, multiply that by the maximum
       phase shift value */
    ftemp = ftemp * (float)phasemax;
    /* store that as the next step in the profile */
    profile[i] = (int)ftemp;
	/* send the profile value to the LPS device */
	status = fnLPS_SetProfileElement(deviceID, i, profile[i]);
	if (status != STATUS_OK) printf ("SetProfileElement returned %s \r\n", fnLPS_perror(status));
    /* get ready for the next one */
    fi = fi + fstep;
  }
  // send the final count to the LPS device
  	status = fnLPS_SetProfileCount(deviceID, nsegs);
	if (status != STATUS_OK) printf ("SetProfileCount returned %s \r\n", fnLPS_perror(status));
}

void profileTriangle(DEVID deviceID, int phasemax) {
  /* calculate values for a triangle phase shift profile. Use the size of
     the 'profile' array and divide a full wave into that many segments. */
  int i, nsegs;
  float fi, fstep;
  float ftemp;
  LPSTATUS status;
  
  nsegs = VNX_PROFILE_LENGTH;
  printf("Making a triangle wave in %d segs\r\n", nsegs);
  /* the wave really has 4 parts - up to max, down to 0, down to min, up to 0
     so we'll divide into 4 pieces and then 'bounce' off of the extremes */
  fstep = 4.0 / (float)nsegs;
  fi = 0.0;
  for (i=0; i<nsegs; i++) {
    ftemp = (1.0 + fi) / 2.0;
    /* and now that we have a 0-1 value, multiply that by the maximum
       phase value */
    ftemp = ftemp * (float)phasemax;
    /* store that as the next step in the profile */
    profile[i] = (int)ftemp;
	/* send the profile value to the LPS device */
	status = fnLPS_SetProfileElement(deviceID, i, profile[i]);
	if (status != STATUS_OK) printf ("SetProfileElement returned %s \r\n", fnLPS_perror(status));
	/* get ready for the next one */
    fi = fi + fstep;
    if (fi >= 1.0) {
      fi = 1.0;
      fstep = -fstep;
    }
    if (fi <= -1.0) {
      fi = -1.0;
      fstep = -fstep;
    }
  }
  // send the final count to the LPS device
  status = fnLPS_SetProfileCount(deviceID, nsegs);
  if (status != STATUS_OK) printf ("SetProfileCount returned %s \r\n", fnLPS_perror(status));  
}


/* displays the profile data as a cheesy graph on the terminal output */
void profileShow(int height) {
  int i, j;
  int rl, rh, rs;

  rs = 360 / height;
  rh = 360;
  rl = rh - rs + 1;
  for (i=height; i>0; i--) {
    for (j=0; j<VNX_PROFILE_LENGTH; j++) {
      if ((profile[j] >= rl) && (profile[j] <= rh))
	printf("*");
      else
	printf(" ");
    }
    printf("\r\n");
    rh = rh - rs;
    rl = rl - rs;
    if (rl < rs) rl = 0;
  }
}

